using System;
using System.Collections;
using gov.va.med.vbecs.Common;

using gov.va.med.vbecs.DAL.VistALink.Client;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>1/22/2004</CreationDate>
	///	<Note>
	///		The Food and Drug Administration classifies this software as a medical device.  
	///		As such, it may not be changed in any way. Modifications to this software may result 
	///		in an adulterated medical device under 21CFR820, the use of which is considered to be 
	///		a violation of US Federal Statutes.  Acquiring and implementing this software through 
	///		the Freedom of information Act requires the implementor to assume total responsibility 
	///		for the software, and become a registered manufacturer of a medical device, 
	///		subject to FDA regulations.
	///	</Note>
	///<summary>Used by the VBECS to control communication with VistA systems via VistALink.</summary>

	#endregion

	public class VistALink
	{
		private const string VBECS_RPC_SECURITY_CONTEXT = "VBECS VISTALINK CONTEXT"; 

		private static VbecsRpcBroker _rpcBroker;
		private static bool _initializedFlag;
		private static IVistALinkUIBinder _uiBinder;

		private static ServerConnectionInfo _primaryVistAServer;

		private VistALink() {} // Must not be instantiated

		static VistALink() {}  // Getting rid of "beforefieldinit" attribute

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6014"> 
		///		<ExpectedInput>None, VistALink is not yet initialized.</ExpectedInput>
		///		<ExpectedOutput>None, VistALink is initialized.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6015"> 
		///		<ExpectedInput>VistALink is already initialized.</ExpectedInput>
		///		<ExpectedOutput>InvalidOperationException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Shortcut method initializing VistALink and setting primary server. 
		/// See description for overload of see cref="SetPrimaryServer"/>.
		/// </summary>
		/// <param name="primaryVistAServer">Primary VistA server passed to overload.</param>
		/// <param name="uiBinder">VistALink UI binder passed to overload.</param>
		/// <param name="vbecsUserDuz">Vbecs user DUZ passed to overload.</param>
		public static void Initialize( ServerConnectionInfo primaryVistAServer, IVistALinkUIBinder uiBinder, string vbecsUserDuz ) 
		{
			Initialize( uiBinder, vbecsUserDuz );
			SetPrimaryServer( primaryVistAServer );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3704"> 
		///		<ExpectedInput>Valid data, VistALink is not yet initialized.</ExpectedInput>
		///		<ExpectedOutput>None, VistALink is initialized.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3705"> 
		///		<ExpectedInput>Valid data, VistALink is already initialized.</ExpectedInput>
		///		<ExpectedOutput>InvalidOperationException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3712"> 
		///		<ExpectedInput>Null uiBinder parameter.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Initializes VistALink communication framework. Binds UI methods to the VistALink code. 
		/// </summary>
		/// <param name="uiBinder">Mediator object tying up VistALink connectivity layer and UI together.</param>
		/// <param name="vbecsUserDuz">
		///		VBECS user VistA DUZ used to perform restrictive check allowing user to access 
		///		VistA system with his/her own credentials only.
		///		If VistA user DUZ is unknown (when using VistALink from MUCs), null may be supplied. 
		///		In this case, DUZ check not performed. 
		///	</param>
		public static void Initialize( IVistALinkUIBinder uiBinder, string vbecsUserDuz ) 
		{
			if( uiBinder == null )
				throw( new ArgumentNullException( "uiBinder" ) );

			if( _initializedFlag )
				throw( new InvalidOperationException( StrRes.SysErrMsg.VistALink.VistaLinkIsAlreadyInitialized().ResString ) );
		
			_rpcBroker = new VbecsRpcBroker( uiBinder, vbecsUserDuz );
			_rpcBroker.ReuseAccessAndVerifyCodes = true;

			_uiBinder = uiBinder;
			_initializedFlag = true;
			
			_primaryVistAServer = null;
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6016"> 
		///		<ExpectedInput>Server connection info pointing to localhost.</ExpectedInput>
		///		<ExpectedOutput>VistALink is unavailable.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6018"> 
		///		<ExpectedInput>Server same as current.</ExpectedInput>
		///		<ExpectedOutput>Nothing is affected.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6021"> 
		///		<ExpectedInput>Valid VistA Server connection info.</ExpectedInput>
		///		<ExpectedOutput>Current connection dropped.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6017"> 
		///		<ExpectedInput>Null</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sets primary server to be used in the system. Disconnects existing connection. 
		/// Will not affect anything if new server is the same as current. 
		/// </summary>
		/// <param name="vistAServer">VistA server connection info to make primary.</param>
		public static void SetPrimaryServer( ServerConnectionInfo vistAServer )
		{
			if( vistAServer == null )
				throw( new ArgumentNullException( "primaryVistAServer" ) );

			EnsureInitialization();

			if( vistAServer.Equals( _primaryVistAServer ) )
				return;

			Disconnect();

			_primaryVistAServer = vistAServer;
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3702"> 
		///		<ExpectedInput>None, VistALink is not connected.</ExpectedInput>
		///		<ExpectedOutput>None, VistALink is connected.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3703"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Establishes connection to currently set VistA server.
		/// </summary>
		/// <returns>True if VistALink is ready to use as a result of operation. Otherwise - false.</returns>
		public static bool Connect()
		{
            // CR3414: remove exception and make it return "false".
		    if (_primaryVistAServer == null)
		        return false;
			return Connect( _primaryVistAServer );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5211"> 
		///		<ExpectedInput>Valid server connection info.</ExpectedInput>
		///		<ExpectedOutput>True.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5212"> 
		///		<ExpectedInput>Non-existent server connection info.</ExpectedInput>
		///		<ExpectedOutput>False.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Establishes connection to VistA server with a given connection info.
		/// </summary>
		/// <param name="server">Server connection info.</param>
		/// <returns>True if VistALink is ready to use as a result of operation. Otherwise - false.</returns>
		public static bool Connect( ServerConnectionInfo server )
		{
			try
			{
				return UnsafeConnect( server );
			}
			catch( VistALinkNetworkErrorException ) 		
			{
				// This is that bad because .NET FCL classes do not provide another way of doing so
				return false;
			}
		}

		/// <summary>
		/// Tries establishing connection to VistA server with a given connection info. 
		/// Will throw exception on any error and return false if user cancels logon.
		/// </summary>
		/// <param name="server">Server to connect to.</param>
		/// <returns>False if user cancels logon. True if logon is successful.</returns>
		internal static bool UnsafeConnect( ServerConnectionInfo server )
		{
			if( server == null )
				throw( new ArgumentNullException( "server" ) );

			EnsureInitialization();

			if( IsAvailable )
				Disconnect();

			return _rpcBroker.Logon( server );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1331"> 
		///		<ExpectedInput>None, VistALink initialized, but not connected.</ExpectedInput>
		///		<ExpectedOutput>None, VistALink is not connected.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1409"> 
		///		<ExpectedInput>None, VistALink is connected.</ExpectedInput>
		///		<ExpectedOutput>None, VistALink is connected.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Changes division on VistA side by logging off and back on.
		/// </summary>
		public static void ChangeDivision()
		{
			EnsureInitialization();

			if( !IsAvailable )
				return;
			
			_rpcBroker.ChangeDivision();
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/29/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3709"> 
		///		<ExpectedInput>None, VistALink is connected.</ExpectedInput>
		///		<ExpectedOutput>None, VistALink is not connected.</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3708"> 
		///		<ExpectedInput>None, VistALink is not connected.</ExpectedInput>
		///		<ExpectedOutput>None, VistALink is not connected.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Disconnects from VistA system. <see cref="VistALink"/> does not have to be initialized to call this method.
		/// </summary>
		public static void Disconnect()
		{		
			// Initialization is not checked to make sure there will not be 
			// an exception in case of VBECS early post-failure cleanup.	

			if( _rpcBroker != null )
				_rpcBroker.Close();
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/29/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3711"> 
		///		<ExpectedInput>None, VistALink is available.</ExpectedInput>
		///		<ExpectedOutput>True.</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3710"> 
		///		<ExpectedInput>None, VistALink is not available.</ExpectedInput>
		///		<ExpectedOutput>False.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates whether VistALink connection is available.
		/// <see cref="VistALink"/> does not have to be initialized to call this property.
		/// </summary>
		public static bool IsAvailable
		{
			get
			{
				// Initialization is not checked to make sure there will not be 
				// an exception in caused by UI event handlers.

				return _initializedFlag && _rpcBroker != null && _rpcBroker.IsLoggedOn;
			}
		}
		
		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/23/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6034"> 
		///		<ExpectedInput>Adding valid handler, VistALink is not connected.</ExpectedInput>
		///		<ExpectedOutput>Handler added, no exception is thrown. Handler fires when VistALink connects.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6035"> 
		///		<ExpectedInput>Removing handler. VistALink is not connected.</ExpectedInput>
		///		<ExpectedOutput>Handler removed, no exception is thrown. Handler doesn't fire when VistALink connects.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6036"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Connection state change event. Fires whenever VistALink goes online or offline.
		/// </summary>
		public static event VistALinkConnectionStateChangedDelegate ConnectionStateChanged
		{
			add
			{
				EnsureInitialization();
				_rpcBroker.ConnectionStateChanged += value;
			}
			remove
			{
				EnsureInitialization();
				_rpcBroker.ConnectionStateChanged -= value;
			}
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/23/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6031"> 
		///		<ExpectedInput>VistALink is connected.</ExpectedInput>
		///		<ExpectedOutput>True.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6032"> 
		///		<ExpectedInput>VistALink is not connected.</ExpectedInput>
		///		<ExpectedOutput>False.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6033"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks VistALink status by sending heartbeat message. 
		/// Basically, works like 'ping'. Does not attempt to restore connection.
		/// 
		/// This is a part of multiclass implementation of the BR_103.01 and BR_103.02.
		/// </summary>
		/// <returns>True if VistALink is available. Otherwise - false.</returns>
		public static bool CheckStatus()
		{
			EnsureInitialization();

			return IsAvailable && _rpcBroker.CheckHeartbeatGetStatus();
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/23/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6028"> 
		///		<ExpectedInput>VistALink not connected.</ExpectedInput>
		///		<ExpectedOutput>VistALink is connected, no actor interaction.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6029"> 
		///		<ExpectedInput>VistALink is not connected, server set to localhost.</ExpectedInput>
		///		<ExpectedOutput>VistALink is not connected, no actor interaction.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6030"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks VistALink status by sending heartbeat message (works like 'ping'). 
		/// If VistALink is not available, tries to reconnect. Does not 
		/// perform any user interaction (unless error occurs). 
		/// 
		/// This is a part of multiclass implementation of the BR_103.01 and BR_103.02.
		/// </summary>
		/// <returns>
		///		True if VistALink was finally acknowledged to be running. 
		///		Otherwise - if reconnect attempt failed - returns false.
		///	</returns>
		public static bool SilentCheckStatusReconnectIfNeeded()
		{
			// Initialization is verified at CheckStatus() call below.

			if( CheckStatus() )
				return true;

			return VistALink.Connect();
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/23/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6025"> 
		///		<ExpectedInput>VistALink is not connected.</ExpectedInput>
		///		<ExpectedOutput>True, VistALink is connected.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6026"> 
		///		<ExpectedInput>VistALink is not connected, server is set to localhost.</ExpectedInput>
		///		<ExpectedOutput>False, VistALink is not connected, assertion exception is thrown (imitation of notification).</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6027"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks VistALink status by sending heartbeat message (works like 'ping'). 
		/// If VistALink is not available, invokes (via supplied UI binder) user interaction
		/// and offers to to reconnect. 
		/// If user cancels reconnect attempt or the latter is not successful, displays 
		/// a message saying that requested operation will be cancelled and returns false.
		/// 
		/// This is a part of multiclass implementation of the BR_103.01 and BR_103.02.
		/// </summary>
		/// <returns>Resulting connection status.</returns>
		public static bool EnsureAvailability()
		{
			if( EnsureAvailabilityWithoutFailureNotification() )
				return true;

			_uiBinder.NotifyAboutProblem( StrRes.SysErrMsg.UC018.VistALinkIsDown().ResString );

			return false;
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/23/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6022"> 
		///		<ExpectedInput>VistALink is not connected.</ExpectedInput>
		///		<ExpectedOutput>True, VistALink is connected, actor interaction was detected.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6023"> 
		///		<ExpectedInput>VistALink is not connected, server is set to localhost.</ExpectedInput>
		///		<ExpectedOutput>False, VistALink is not connected, no exception is thrown (imitation of notification), actor interaction was detected.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6024"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks VistALink status by sending heartbeat message (works like 'ping'). 
		/// If VistALink is not available, invokes (via supplied UI binder) user interaction
		/// and offers to to reconnect. 
		/// If user cancels reconnect attempt or the latter is not successful returns false.
		/// 
		/// This is a part of multiclass implementation of BR_103.01.
		/// </summary>
		/// <returns>Resulting connection status.</returns>
		public static bool EnsureAvailabilityWithoutFailureNotification()
		{
			EnsureInitialization();
			
			return _uiBinder.CheckVistALinkReconnectIfNeeded();
		}

		/// <summary>
		/// Checks if VistALink has been initialized. Throws <see cref="InvalidOperationException"/> if it hasn't.
		/// </summary>
		private static void EnsureInitialization()
		{
			if( !_initializedFlag )
				throw( new InvalidOperationException( StrRes.SysErrMsg.VistALink.VistaLinkIsNotInitialized().ResString ) );
		}

		/// <summary>
		/// Creates <see cref="RpcRequest"/> with preset VBECS RPC security context.
		/// </summary>
		/// <param name="rpcRequestName">RPC request name.</param>
		/// <returns>New RPC request with a given name and a preset security context.</returns>
		internal static RpcRequest CreateRpcRequest( string rpcRequestName )
		{
			if( rpcRequestName == null )
				throw( new ArgumentNullException( "rpcRequestName" ) );

			return new RpcRequest( rpcRequestName, VBECS_RPC_SECURITY_CONTEXT );
		}

	    /// <summary>
	    /// Creates <see cref="RpcRequest"/> with preset VBECS RPC security context.
        /// CR 3304
	    /// </summary>
	    /// <param name="rpcRequestName">RPC request name.</param>
        /// <param name="rpcTimeout">RPC timeout. If its null, it will be initialized with default value from GlobalConfig.</param>
	    /// <returns>New RPC request with a given name and a preset security context.</returns>
        internal static RpcRequest CreateRpcRequest(string rpcRequestName, Timeout rpcTimeout)
        {
            if (rpcRequestName == null)
                throw (new ArgumentNullException("rpcRequestName"));

            return new RpcRequest(rpcRequestName, VBECS_RPC_SECURITY_CONTEXT, rpcTimeout);
        }

		/// <summary>
		/// Gets a reference to a <see cref="VbecsRpcBroker"/> used to communicate with VistA system.
		/// Does not check whether VistALink is connected or not. Caller is expected to ensure connection 
		/// availability, if needed.
		/// </summary>
		/// <returns>Reference to <see cref="VbecsRpcBroker"/>.</returns>
		internal static VbecsRpcBroker GetRpcBroker()
		{
			EnsureInitialization();

			return _rpcBroker;
		}
	}
}
